//
// C/C++ headers
//
#include <fstream>
#include <string.h>
//
// ROOT headers
//
#include <TTree.h>
#include <TClassEdit.h>
#include <TObject.h>
#include <TList.h>
#include <TArrayL.h>
#include <TSystem.h>
#include <TSystemDirectory.h>
#include <TString.h>
#include <TFile.h>
#include <TClass.h>
#include <TCanvas.h>
#include <TH1.h>
#include <TH1F.h>
#include <TH2D.h>
#include <TLatex.h>
#include <TPad.h>
#include <TSQLServer.h>
#include <TSQLRow.h>
#include <TSQLResult.h>
#include <TClonesArray.h>
//
// YODA headers
//
#include <PamelaRun.h>
#include <RegistryEvent.h>
#include <physics/trigger/TriggerEvent.h>
#include <physics/tof/TofEvent.h>
//
// RunInfo header
//
#include <RunInfo.h>
//
// This program headers
//
#include <ToFCore.h>
#include <ToFLevel2.h>
#include <ToFVerl2.h>
//
//
// Declaration of the core fortran routines
//
#define tofl2com tofl2com_
extern "C" int tofl2com();
#define toftrk toftrk_
extern "C" int toftrk();
#define rdtofcal rdtofcal_
extern "C" int rdtofcal(char []);

//
// Tracker classes headers and definitions
//
#include <TrkLevel2.h>
//
using namespace std;
//
//
// CORE ROUTINE
//
//
int ToFCore(ULong64_t run, TFile *file, TSQLServer *dbc, Int_t ToFargc, char *ToFargv[]){
  //
  //
  // Set these to true to have a very verbose output.
  //
  Bool_t verbose = false;
  Bool_t debug = false;
  //
  TString processFolder = "ToFFolder";
  if ( ToFargc > 0 ){
    Int_t i = 0;
    while ( i < ToFargc ){
      if ( !strcmp(ToFargv[i],"-processFolder") ) {
	if ( ToFargc < i+1 ){
	  throw -3;
	};
	processFolder = (TString)ToFargv[i+1];
	i++;
      };
      if ( !strcmp(ToFargv[i],"-v") ||  !strcmp(ToFargv[i],"--verbose") ) {
	verbose = true;
      };
      if ( !strcmp(ToFargv[i],"-g") ||  !strcmp(ToFargv[i],"--debug") ) {
	debug = true;
      };
      i++;
    };
  };
  //
  //
  // Output directory is the working directoy.
  //
  const char* outdir = gSystem->DirName(gSystem->DirName(file->GetPath()));
  //
  // Variables for level2 
  //
  TTree *tracker = 0;
  TTree *toft = 0;
  Long64_t nevents = 0LL;
  //
  // variables needed to reprocess data
  //
  TString tofversion;
  ItoRunInfo *runinfo = 0;
  TArrayL *runlist = 0;
  TTree *toftclone = 0;
  Bool_t reproc = false;
  Bool_t reprocall = false;
  UInt_t nobefrun = 0;
  UInt_t noaftrun = 0;
  UInt_t numbofrun = 0;
  stringstream ftmpname;
  TString fname;
  Long64_t totfileentries = 0ULL;
  Long64_t idRun = 0LL;
  //
  // variables needed to handle error signals
  //
  Int_t code = 0;
  Int_t sgnl;
  //
  // tof level2 classes
  //
  ToFLevel2 *tof = new ToFLevel2();
  ToFLevel2 *tofclone = new ToFLevel2();
  //
  // tracker level2 variables
  //
  TrkLevel2 *trk = new TrkLevel2();
  Long64_t nevtrkl2 = 0;
  //
  // define variables for opening and reading level0 file
  //
  TFile *l0File = 0;
  TTree *l0tr = 0;
  TBranch *l0head = 0;
  TBranch *l0registry = 0;
  TBranch *l0trig = 0;
  TBranch *l0tof = 0;
  pamela::RegistryEvent *l0reg=0;
  pamela::trigger::TriggerEvent *trig = 0;
  pamela::tof::TofEvent *tofEvent = 0;
  //
  // Define other basic variables
  // 
  UInt_t procev = 0;
  stringstream file2;
  stringstream file3;
  stringstream qy;
  Int_t itr = -1;    
  Int_t totevent = 0;
  ULong64_t atime = 0ULL;
  Int_t ei = 0; 
  Int_t re = 0;
  //
  // Working filename
  //
  TString outputfile;
  stringstream name;
  name.str("");
  name << outdir << "/";
  //
  // temporary file and folder
  //
  TFile *tempfile = 0;
  TTree *temptof = 0;
  stringstream tempname;
  stringstream toffolder;
  tempname.str("");
  tempname << outdir;
  tempname << "/" << processFolder.Data();
  toffolder.str("");
  toffolder << tempname.str().c_str();
  gSystem->MakeDirectory(toffolder.str().c_str());
  tempname << "/toftree_run";
  tempname << run << ".root";  
  //
  // variables needed to load magnetic field maps
  // 
  Int_t ntrkentry = 0;
  ULong64_t tttrkpar1 = 0ULL;
  Bool_t trkpar1 = true;
  ULong64_t tttofpar1 = 0ULL;
  Bool_t tofpar1 = true;
  //
  // DB classes
  //
  GL_ROOT *glroot = new GL_ROOT();
  GL_PARAM *glparam = new GL_PARAM();
  //
  // declaring external output and input structures
  //
  extern struct ToFInput  tofinput_;
  extern struct ToFOutput tofoutput_;
  //
  // Let's start!
  //
  //
  // As a first thing we must check what we have to do: if run = 0 we must process all events in the file has been passed
  // if run != 0 we must process only that run but first we have to check if the tree ToF already exist in the file
  // if it exists we are reprocessing data and we must delete that entries, if not we must create it.
  // 
  if ( run == 0ULL )  reproc = true;
  //
  //
  // Output file is "outputfile"
  //
  if ( !file->IsOpen() ){
    if ( verbose ) printf(" ToF - ERROR: cannot open file for writing\n");
    throw -301;    
  };
  //
  // Does it contain the Tracker tree?
  //
  tracker = (TTree*)file->Get("Tracker");
  if ( !tracker ) {
    if ( verbose ) printf(" TOF - ERROR: no tracker tree\n");
    code = -302;
    goto closeandexit;
  };
  //
  // get tracker level2 data pointer
  //
  tracker->SetBranchAddress("TrkLevel2",&trk);  
  nevtrkl2 = tracker->GetEntries();
  //
  // Retrieve GL_RUN variables from the level2 file
  //  
  tofversion = ToFInfo(false); // we should decide how to handle versioning system
  //
  // create an interface to RunInfo called "runinfo"
  //
  // ItoRunInfo= interface with RunInfo and GL_RUN
  runinfo = new ItoRunInfo(file);
  //
  // open "Run" tree in level2 file, if not existing return an error (sngl != 0)
  //
  sgnl = 0;
  sgnl = runinfo->Update(run, "TOF",tofversion);
  if ( sgnl ){
    if ( verbose ) printf(" TOF - ERROR: RunInfo exited with non-zero status\n");
    code = sgnl;
    goto closeandexit;
  } else {
    sgnl = 0;
  };
  //
  // number of events in the file BEFORE the first event of our run
  //
  nobefrun = runinfo->GetFirstEntry();
  //
  // total number of events in the file 
  //
  totfileentries = runinfo->GetFileEntries();
  //
  // first file entry AFTER the last event of our run
  //
  noaftrun = runinfo->GetLastEntry() + 1;
  //
  // number of run to be processed
  //
  numbofrun = runinfo->GetNoRun();
  //
  // Try to access the ToF tree in the file, if it exists we are reprocessing data if not we are processing a new run
  //
  toftclone = (TTree*)file->Get("ToF");
  //
  if ( !toftclone ){
    //
    // tree does not exist, we are not reprocessing
    //
    reproc = false;
    if ( run == 0ULL && verbose ) printf(" ToF - WARNING: you are reprocessing data but ToF tree does not exist!\n");
    if ( runinfo->IsReprocessing() && run != 0ULL && verbose ) printf(" ToF - WARNING: it seems you are not reprocessing data but ToF\n versioning information already exists in RunInfo.\n");

  } else {
    //
    // tree exists, we are reprocessing data. Are we reprocessing a single run or all the file?
    //
    reproc = true;
    //
    // update versioning information
    //
    if ( verbose ) printf("\n Preparing the pre-processing...\n");
    //
    if ( run == 0ULL ){
      //
      // we are reprocessing all the file
      // if we are reprocessing everything we don't need to copy any old event and we can just work with the new tree and delete the old one immediately
      //
      reprocall = true;
      //
      if ( verbose ) printf("\n ToF - WARNING: Reprocessing all runs\n");
      //
    } else {
      //
      // we are reprocessing a single run, we must copy to the new tree the events in the file which preceed the first event of the run
      //
      reprocall = false;
      //
      if ( verbose ) printf("\n ToF - WARNING: Reprocessing run number %llu \n",run);
      //
      // copying old tree to a new file
      //
      tempfile = new TFile(tempname.str().c_str(),"RECREATE");
      temptof = toftclone->CloneTree(-1,"fast");
      temptof->SetName("ToF-old");
      tempfile->Write();
      tempfile->Close();  
    }
    //
    // Delete the old tree from old file and memory
    //
    toftclone->Delete("all");
    //
    if ( verbose ) printf(" ...done!\n");
    //
  };
  //
  // create ToF detector tree toft
  // 
  file->cd();
  toft = new TTree("ToF-new","PAMELA Level2 ToF data");
  toft->Branch("ToFLevel2","ToFLevel2",&tof);
  //
  if ( reproc && !reprocall ){
    //
    //  open new file and retrieve all tree informations
    //
    tempfile = new TFile(tempname.str().c_str(),"READ");
    toftclone = (TTree*)tempfile->Get("ToF-old");
    toftclone->SetBranchAddress("ToFLevel2",&tofclone);
    //      
    if ( nobefrun > 0 ){
      if ( verbose ) printf("\n Pre-processing: copying events from the old tree before the processed run\n");   
      if ( verbose ) printf(" Copying %u events in the file which are before the beginning of the run %llu \n",nobefrun,run);
      if ( verbose ) printf(" Start copying at event number 0, end copying at event number %u \n",nobefrun);
      for (UInt_t j = 0; j < nobefrun; j++){
	//
	toftclone->GetEntry(j);	  
	//
	// copy tofclone to tof
	//
	tof = new ToFLevel2();
	memcpy(&tof,&tofclone,sizeof(tofclone));
	//
	// Fill entry in the new tree
	//
	toft->Fill();
	//
      };
      if ( verbose ) printf(" Finished successful copying!\n");
    };          
  };
  //
  // Get the list of run to be processed, if only one run has to be processed the list will contain one entry only.
  //
  runlist = runinfo->GetRunList();
  //
  // Loop over the run to be processed
  //
  for (UInt_t irun=0; irun < numbofrun; irun++){
    //
    // retrieve the first run ID to be processed using the RunInfo list
    //
    idRun = runlist->At(irun);
    if ( verbose ) printf("\n\n\n ####################################################################### \n");
    if ( verbose ) printf("                    PROCESSING RUN NUMBER %i \n",(int)idRun);
    if ( verbose ) printf(" ####################################################################### \n\n\n");
    //
    runinfo->ID_REG_RUN = 0ULL;
    //
    // store in the runinfo class the GL_RUN variables for our run
    //
    sgnl = 0;
    sgnl = runinfo->GetRunInfo(idRun);
    if ( sgnl ){
      if ( verbose ) printf(" TOF - ERROR: RunInfo exited with non-zero status\n");
      code = sgnl;
      goto closeandexit;
    } else {
      sgnl = 0;
    };
    //
    // now you can access that variables using the RunInfo class this way runinfo->ID_REG_RUN
    //
    if ( runinfo->ID_REG_RUN == 0 ){
      if ( verbose ) printf("\n TOF - ERROR: no run with ID_RUN = %i \n\n Exiting... \n\n",(int)idRun);
      code = -5;
      goto closeandexit;    
    };
    //
    // Search in the DB the path and name of the LEVEL0 file to be processed.
    //
    glroot->Query_GL_ROOT(runinfo->ID_REG_RUN,dbc);
    //
    ftmpname.str("");
    ftmpname << glroot->PATH.Data() << "/";
    ftmpname << glroot->NAME.Data();
    fname = ftmpname.str().c_str();
    //
    // print out informations
    //
    totevent = runinfo->EV_REG_PHYS_TO - runinfo->EV_REG_PHYS_FROM + 1;
    if ( verbose ) printf("\n LEVEL0 data file: %s \n",fname.Data());
    if ( verbose ) printf(" RUN HEADER absolute time is:  %llu \n",runinfo->RUNHEADER_TIME);
    if ( verbose ) printf(" RUN TRAILER absolute time is: %llu \n",runinfo->RUNTRAILER_TIME);
    if ( verbose ) printf(" %i events to be processed for run %llu: from %i to %i (reg entries)\n\n",totevent,idRun,runinfo->EV_REG_PHYS_FROM,runinfo->EV_REG_PHYS_TO);
    //
    // Open Level0 file
    //
    l0File = new TFile(fname.Data());
    if ( !l0File ) {
      if ( verbose ) printf(" TOF - ERROR: problems opening  Level0 file\n");
      code = -6;
      goto closeandexit;
    };
    l0tr = (TTree*)l0File->Get("Physics");
    if ( !l0tr ) {
      if ( verbose ) printf(" TOF - ERROR: no Physics tree in Level0 file\n");
      l0File->Close();
      code = -7;
      goto closeandexit;
    };
    l0head = l0tr->GetBranch("Header");
    if ( !l0head ) {
      if ( verbose ) printf(" TOF - ERROR: no Header branch in Level0 tree\n");
      l0File->Close();
      code = -8;
      goto closeandexit;    
    };
    l0registry = l0tr->GetBranch("Registry");
    if ( !l0registry ) {
      if ( verbose ) printf(" TOF - ERROR: no Registry branch in Level0 tree\n");
      l0File->Close();
      code = -9;
      goto closeandexit;    
    };
    l0trig = l0tr->GetBranch("Trigger");
    if ( !l0trig ) {
      if ( verbose ) printf(" TOF - ERROR: no Trigger branch in Level0 tree\n");
      l0File->Close();
      code = -300;
      goto closeandexit;
    };
    l0tof = l0tr->GetBranch("Tof");
    if ( !l0tof ) {
      if ( verbose ) printf(" TOF - ERROR: no ToF branch in Level0 tree\n");
      l0File->Close();
      code = -303;
      goto closeandexit;
    };
    //
    l0tr->SetBranchAddress("Trigger", &trig);
    l0tr->SetBranchAddress("Tof", &tofEvent);
    l0tr->SetBranchAddress("Registry", &l0reg);
    //
    nevents = l0registry->GetEntries();
    //
    if ( nevents < 1 ) {
      if ( verbose ) printf(" TOF - ERROR: Level0 file is empty\n\n");
      l0File->Close();
      code = -11;
      goto closeandexit;
    };
    //
    if ( runinfo->EV_REG_PHYS_TO > nevents-1 ) {
      if ( verbose ) printf(" TOF - ERROR: too few entries in the registry tree\n");
      l0File->Close();
      code = -12;
      goto closeandexit;
    };
    //
    // Check if we have to load parameter files (or calibration associated to runs and not to events)
    //
    // for example let's assume that we could have different magnetic field maps for different runs:
    //
    if ( trkpar1 || ( tttrkpar1 != 0 && tttrkpar1 < runinfo->RUNHEADER_TIME ) ){
      trkpar1 = false;
      // read from DB infos about Magnetic filed maps
      glparam->Query_GL_PARAM(runinfo->RUNHEADER_TIME,"field",dbc); // parameters stored in DB in GL_PRAM table
      tttrkpar1 = glparam->TO_TIME;  
      // ----------------------------
      // Read the magnetic field
      // ----------------------------
      if ( verbose ) printf(" Reading magnetic field maps: \n");
      trk->LoadField(glparam->PATH+glparam->NAME);
      if ( verbose ) printf("\n");
    };
    // Bug Found: Emiliano 30 May 2006
    //     if ( tofpar1 || ( tttofpar1 != 0 && tttofpar1 < runinfo->RUNHEADER_TIME ) ){
    //       tofpar1 = false;
    //       //
    //       glparam->Query_GL_PARAM(runinfo->RUNHEADER_TIME,"tof param",dbc); // parameters stored in DB in GL_PRAM table
    //       //
    //       tttofpar1 = glparam->TO_TIME;  
    //       rdtofcal((char *)(glparam->PATH+glparam->NAME).Data());
    //     };
    // Corrected:
    if ( tofpar1 || ( tttofpar1 != 0 && tttofpar1 < runinfo->RUNHEADER_TIME ) ){
      tofpar1 = false;
      //
      Int_t error=glparam->Query_GL_PARAM(runinfo->RUNHEADER_TIME,"tof param",dbc); // parameters stored in DB in GL_PRAM table
      if ( error<0 ) {
	code = error;
	goto closeandexit;
      };
      //
      tttofpar1 = glparam->TO_TIME;  
      rdtofcal((char *)(glparam->PATH+glparam->NAME).Data());
    };
    //
    // run over all the events of the run
    //
    if ( verbose ) printf("\n Ready to start! \n\n Processed events: \n\n");
    //
    for ( re = runinfo->EV_REG_PHYS_FROM; re <= runinfo->EV_REG_PHYS_TO; re++){
      //
      if ( procev%1000 == 0 && procev > 0 && verbose ) printf(" %iK \n",procev/1000);	
      //
      l0registry->GetEntry(re);
      //
      // absolute time of this event
      //
      atime = l0reg->absTime;
      //
      // physics events is at entry number ei where
      //
      ei = l0reg->event;
      //
      // paranoid check
      //
      if ( atime > runinfo->RUNTRAILER_TIME || atime < runinfo->RUNHEADER_TIME  ) {
	if ( verbose ) printf(" TOF - WARNING: event at time outside the run time window, skipping it\n");
	goto jumpev;
      };
      //
      // retrieve tracker informations, the LEVEL2 entry which correspond to our event will be "itr"
      //
      if ( !reprocall ){
	itr = nobefrun + (re - runinfo->EV_REG_PHYS_FROM);
      } else {
	itr = runinfo->GetFirstEntry() + (re - runinfo->EV_REG_PHYS_FROM);
      };
      if ( itr > nevtrkl2 ){  // nevtrkl2 tracker entry number
	if ( verbose ) printf(" TOF - ERROR: no tracker events with entry = %i in Level2 file\n",itr);
	l0File->Close();
	code = -313;
	goto closeandexit; 
      };
      tracker->GetEntry(itr); 
      ///
      l0tof->GetEntry(ei);
      l0trig->GetEntry(ei);
      ///
      //
      procev++;
      //
      // start processing
      //
      tof = new ToFLevel2();

      //
      // Here we will use some procedure to calibrate our data and put some kind of informations in the cinput structure

      for (Int_t gg=0; gg<4;gg++){
	for (Int_t hh=0; hh<12;hh++){
	  tofinput_.tdc[hh][gg]=tofEvent->tdc[gg][hh];          
	  tofinput_.adc[hh][gg]=tofEvent->adc[gg][hh];
	};	    
      };

      for (Int_t hh=0; hh<5;hh++){
	tofinput_.patterntrig[hh]=trig->patterntrig[hh];
      };	    
      //
      // Here we have calibrated data, ready to be passed to the FORTRAN routine which will extract common and track-related  variables.
      //
      //
      // Calculate tracks informations from ToF alone
      //
      tofl2com();
      //
      // and now we must copy from the output structure to the level2 class:
      //
      // common variables not related to the track
      for (Int_t kk=0; kk<3;kk++){
	tof->xtofpos[kk] = tofoutput_.xtofpos[kk];
	tof->ytofpos[kk] = tofoutput_.ytofpos[kk];
      }

      for (Int_t kk=0; kk<6;kk++){
	tof->tof_i_flag[kk]=tofoutput_.tof_i_flag[kk];
	tof->tof_j_flag[kk]=tofoutput_.tof_j_flag[kk];
      }

      for (Int_t kk=0; kk<13;kk++){
	tof->betatof_a[kk] = tofoutput_.betatof_a[kk];	
      }
      
      for (Int_t hh=0; hh<12;hh++){
	for (Int_t kk=0; kk<4;kk++){
	  tof->adctof_c[hh][kk] = tofoutput_.adctof_c[hh][kk];
	}
      }

      for (Int_t hh=0; hh<12;hh++){
	for (Int_t kk=0; kk<4;kk++){
	  tof->tdc_c[hh][kk] = tofoutput_.tdc_c[hh][kk];
	}
      }
      
      //
      // Calculate track-related variables 
      //
      //
      // Calculate variables related to tracks only if we have at least one track (from selftrigger and/or tracker)
      //
      ntrkentry = 0;
      //
      if ( trk->GetNTracks() > 0 ){
	//
	// We have at least one track
	//
	//
	// Run over tracks
	//
	// example: how to use tracker data
	for(Int_t nt=0; nt < trk->ntrk(); nt++){  
	  //
	  TrkTrack *ptt = trk->GetStoredTrack(nt); 
	  //
	  // Copy the alpha vector in the input structure
	  //
	  for (Int_t e = 0; e < 5 ; e++){
	    tofinput_.al_pp[e] = ptt->al[e];
	  };	  
	  //
	  // Get tracker related variables for this track
	  //
	  toftrk();
	  //
	  // Copy values in the class from the structure (we need to use a temporary class to store variables).
	  //
	  ToFTrkVar *t_tof = new ToFTrkVar();
	  for (Int_t kk=0; kk<13;kk++){
	    t_tof->beta_a[kk] = tofoutput_.beta_a[kk];
	  }
	  
	  for (Int_t hh=0; hh<12;hh++){
	    for (Int_t kk=0; kk<4;kk++){
	      t_tof->adc_c[hh][kk] = tofoutput_.adc_c[hh][kk];
	    }
	  }
	  //
	  // Store the tracker track number in order to be sure to have shyncronized data during analysis
	  //
	  t_tof->trkseqno = nt;
	  //
	  // create a new object for this event with track-related variables
	  //
	  TClonesArray &t = *tof->ToFTrk;
	  new(t[ntrkentry]) ToFTrkVar(*t_tof);
	  //
	  ntrkentry++;	
	  //
	}; // loop on all the tracks
      }; 
      //
      // Fill the rootple 
      //
      toft->Fill();
      //    
      //
    jumpev:
      debug = false;
      //
    };
  }; // process all the runs
  //
  if ( verbose ) printf("\n Finished processing data \n");
  //
 closeandexit:
  //
  // we have finished processing the run(s). If we processed a single run now we must copy all the events after our run from the old tree to the new one and delete the old tree.
  //
  if ( !reprocall && reproc && code >= 0 ){
    if ( totfileentries > noaftrun ){
      if ( verbose ) printf("\n Post-processing: copying events from the old tree after the processed run\n");   
      if ( verbose ) printf(" Copying %i events in the file which are after the end of the run %i \n",(int)(totfileentries-noaftrun),(int)run);
      if ( verbose ) printf(" Start copying at event number %i end copying at event number %i \n",(int)noaftrun,(int)totfileentries);
      for (UInt_t j = noaftrun; j < totfileentries; j++ ){
	//
	// Get entry from old tree
	//
	toftclone->GetEntry(j);
	//
	// copy tofclone to tof
	//
	tof = new ToFLevel2();
	memcpy(&tof,&tofclone,sizeof(tofclone));
	//
	// Fill entry in the new tree
	//
	toft->Fill();
      };
      if ( verbose ) printf(" Finished successful copying!\n");
    };
  };
  //
  // Close files, delete old tree(s), write and close level2 file 
  //
  if ( l0File ) l0File->Close();
  if ( tempfile ) tempfile->Close();            
  gSystem->Unlink(tempname.str().c_str());
  if ( tracker )  tracker->Delete(); // delete tracker tree from memory only to avoid writing a copy to file!
  //
  if ( code < 0 && verbose ) printf("\n TOF - ERROR: an error occurred, try to save anyway...\n");
  if ( verbose ) printf("\n Writing and closing rootple\n");
  if ( runinfo ) runinfo->Close();    
  if ( toft ) toft->SetName("ToF");    
  if ( file ){
    file->cd();
    file->Write("ToF");
  };
  //
  gSystem->Unlink(toffolder.str().c_str());
  //
  // the end
  //
  if ( verbose ) printf("\n Exiting...\n");
  if(toft)toft->Delete();
  if ( code < 0 ) throw code;
  return(code);
}
